// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_IBLUETOOTH_CALLBACKS_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_IBLUETOOTH_CALLBACKS_H

#include "common.h"

/**
 * This class defines the interfaces that are necessary to enable automatic
 * bluetooth pairing via the GAL protocol. You must subclass this class and
 * implement the methods to integrate with your platform. More details are
 * available in the documentation for class BluetoothEndpoint.
 */
class IBluetoothCallbacks {
public:
    virtual ~IBluetoothCallbacks() { }

    /**
     * Once this function is called, the head unit must make itself prepared for
     * Bluetooth pairing which will be initiated by the phone.
     *
     *
     * To be ready for pairing, the head unit must disconnect all the other HFP
     * (Hands-Free Profile) connections, set the GAL-connected phone's priority
     * as the highest and make the car's Bluetooth module discoverable to the
     * phone.
     *
     *
     * When the head unit is ready to be paired, it must call
     * |BluetoothEndpoint::onReadyForPairing|. Then the phone will initiate the
     * Bluetooth pairing-and-connection process with the hewad unit. In this
     * process, pairing step will be skipped if both the phone and the car has
     * link key (pairing information) for each other. In that case the phone
     * will initiate HFP (Hands-Free Profile) connection only.
     *
     *
     * The head unit must be prepared for pairing and call
     * |BluetoothEndpoint::onReadyForPairing|, even if it has already been
     * paired with the phone. This is necessary because the phone may have lost
     * the pairing information, even though the head unit has not. In that case,
     * the phone may initiate pairing.
     *
     *
     * If the head unit cannot be prepared for pairing within 1 second after
     * |onPairingRequest| is called, the head unit should call
     * |BluetoothEndpoint::requestDelayedPairing|. And then the phone will
     * assume that the head unit will call
     * |BluetoothEndpoint::onReadyForPairing| after a while.
     *
     *
     * Even if the phone and the head unit have already been paired after the
     * head unit calls |BluetoothEndpoint::onReadyForPairing|, the phone may
     * unpair and invoke |onPairingRequest| again, in order to fix corrupted
     * state. So the car must be able to handle BluetoothPairingRequest messages
     * multiple times in the same GAL connetion.
     *
     *
     * @param phoneAddress The Bluetooth address of the phone. For example,
     *        "00:11:22:AA:BB:CC"
     * @param pairingMethod The pairing method, as defined in enum
     *        BluetoothPairingMethod in protos.proto.
     */
    virtual void onPairingRequest(const string& phoneAddress,
                                  int pairingMethod) = 0;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_IBLUETOOTH_CALLBACKS_H
